#!/usr/bin/env python3
"""Test script for the enhanced agentic RAG system"""

import sys
import os
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

from main import RAGSystem

def test_enhanced_agent():
    """Test the enhanced agentic behavior"""
    print("Testing Enhanced Agentic RAG System")
    print("=" * 50)

    # Initialize system
    rag_system = RAGSystem()
    doc_count = rag_system.setup_system()

    if doc_count == 0:
        print("No documents available. Please add PDFs to test.")
        return

    print(f"System ready with {doc_count} documents")
    print()

    # Test questions designed to validate improvements
    test_questions = [
        "what cultures were in america before columbus",
        "tell me about columbus and jewish participation",
        "what is the secret destiny of america",
        "who are the nagas",
        "explain esoteric traditions"
    ]

    for i, question in enumerate(test_questions, 1):
        print(f"Question {i}: {question}")
        print("-" * 40)

        try:
            result = rag_system.search_and_answer(question)

            if isinstance(result, str):
                print(f"Answer: {result}")
            else:
                print(f"Answer: {result['answer'][:500]}{'...' if len(result['answer']) > 500 else ''}")
                print(f"Sources: {len(result['sources'])} chunks")

                # Show source diversity
                source_names = set(src['source'] for src in result['sources'])
                print(f"Source documents: {', '.join(source_names)}")

        except Exception as e:
            print(f"Error: {e}")

        print()

def test_agentic_improvements():
    """Test specific agentic improvements"""
    print("Testing Specific Agentic Improvements")
    print("=" * 50)

    rag_system = RAGSystem()
    doc_count = rag_system.setup_system()

    if doc_count == 0:
        return

    # Initialize agent
    rag_system.initialize_model()
    agent = rag_system.rag_agent

    # Test spelling correction
    print("1. Spelling Correction Test:")
    test_text = "what culures were in amerca befor columbis"
    corrected = agent.correct_spelling(test_text)
    print(f"   Input: {test_text}")
    print(f"   Corrected: {corrected}")
    print()

    # Test question analysis
    print("2. Question Analysis Test:")
    questions = [
        "what cultures were in america before columbus",
        "explain the secret destiny of america",
        "compare different esoteric traditions"
    ]
    for q in questions:
        qtype = agent.analyze_question(q)
        print(f"   '{q}' -> {qtype}")
    print()

    # Test fuzzy matching
    print("3. Fuzzy Matching Test:")
    test_pairs = [
        ("america", "americas"),
        ("culture", "cultures"),
        ("columbus", "columbis"),
        ("before", "befor")
    ]
    for w1, w2 in test_pairs:
        score = agent.fuzzy_match_score(w1, w2)
        print(".2f")
    print()

if __name__ == "__main__":
    test_enhanced_agent()
    test_agentic_improvements()
