# NBA Data Integration for Backtesting AI

## Overview

This integration adds comprehensive NBA data access to the backtesting platform, enabling users to build and test sports betting strategies using real NBA statistics, player data, and game information.

## Architecture

### Data Service (`nba_data_service.py`)
- **Python-based NBA API client** using the `nba_api` library
- **Intelligent caching system** to avoid repulling data
- **Multiple data endpoints**: players, teams, games, live games, player stats
- **Command-line interface** for Next.js API integration

### API Endpoints
- `GET /api/nba/players` - NBA player roster data
- `GET /api/nba/teams` - NBA team information
- `GET /api/nba/games` - Historical game data
- `GET /api/nba/games/live` - Live game scores and status
- `GET /api/nba/players/[playerId]/stats` - Individual player statistics

### Database Schema
New Prisma models for storing NBA data:
- `NBAPlayer` - Player information and metadata
- `NBATeam` - Team details and statistics
- `NBAGame` - Game results and schedules
- `NBAPlayerStats` - Detailed player performance data
- `NBATeamStats` - Team performance metrics
- `NBACacheEntry` - Cached API responses

### Frontend Component
- `NBADataViewer` component with tabbed interface
- Real-time data browsing and filtering
- Live game tracking with auto-refresh
- Player statistics visualization

## Features

### ✅ Data Sources
- **NBA Official API**: Official player, team, and game data
- **Live Game Data**: Real-time scores and game status
- **Historical Statistics**: Multi-season player and team performance
- **Game Schedules**: Past and upcoming game information

### ✅ Caching Strategy
- **Smart Caching**: 24-hour cache for static data, 6 minutes for live data
- **Cache Keys**: MD5-hashed endpoint + parameters
- **Cache Statistics**: Hit rates, storage usage, cache management
- **Manual Refresh**: Force fresh data when needed

### ✅ Performance Optimizations
- **Background Processing**: Python service runs asynchronously
- **Database Indexing**: Optimized queries on frequently accessed fields
- **Response Compression**: Efficient data transfer
- **Connection Pooling**: Optimized database connections

## Usage Examples

### Basic Data Access
```typescript
// Get NBA players
const players = await fetch('/api/nba/players?limit=50&cache=true');

// Get teams
const teams = await fetch('/api/nba/teams?cache=true');

// Get games for current season
const games = await fetch('/api/nba/games?season=2023-24&limit=20');

// Get live games
const liveGames = await fetch('/api/nba/games/live?cache=false');

// Get player statistics
const stats = await fetch('/api/nba/players/203999/stats'); // Nikola Jokić
```

### Strategy Integration
```typescript
// Use NBA data in sports betting strategies
const playerData = await fetch('/api/nba/players');
const gameData = await fetch('/api/nba/games');

// Analyze player performance trends
// Build predictive models for game outcomes
// Create betting strategies based on historical data
```

## Cache Management

### Cache Configuration
```python
# In nba_data_service.py
DEFAULT_CACHE_HOURS = {
    "players": 24,      # Static roster data
    "teams": 24,        # Team information
    "games": 6,         # Recent game data
    "live_games": 0.1,  # 6 minutes for live data
    "player_stats": 6,  # Player performance data
}
```

### Cache Statistics
```bash
# Check cache performance
python nba_data_service.py cache_stats '{}'

# Clear specific cache entries
python nba_data_service.py clear_cache '{"pattern": "players"}'
```

## Sports Betting Strategies

### Available Data for Strategies
- **Player Statistics**: Points, rebounds, assists, shooting percentages
- **Team Performance**: Win/loss records, point differentials
- **Game Outcomes**: Historical results and betting lines
- **Live Data**: Real-time scores and game status
- **Season Trends**: Multi-year performance analysis

### Strategy Examples
1. **Player Performance Prediction**: Use historical stats to predict game outcomes
2. **Injury Impact Analysis**: Track player availability and team performance
3. **Home/Away Performance**: Analyze venue-based performance trends
4. **Head-to-Head Matchups**: Historical performance between specific teams
5. **Season Momentum**: Track team performance trends over time

## API Rate Limiting

### NBA.com API Limits
- **Rate Limits**: NBA API has request limits per minute/hour
- **Caching**: Reduces API calls by 80-90% for repeated requests
- **Error Handling**: Graceful fallbacks when API limits are reached

### User Access Control
- **Authentication Required**: All NBA data access requires user login
- **Subscription Tiers**: Different data access levels by subscription
- **Usage Tracking**: Monitor data consumption per user

## Data Quality & Reliability

### Data Validation
- **Schema Validation**: All data validated against expected schemas
- **Error Handling**: Robust error handling for API failures
- **Data Freshness**: Timestamps track data freshness and updates

### Monitoring & Alerts
- **API Health Checks**: Monitor NBA API availability
- **Data Quality Metrics**: Track data completeness and accuracy
- **Performance Alerts**: Notifications for slow response times

## Development & Testing

### Local Development
```bash
# Install Python dependencies
source nba_api_env/bin/activate
pip install nba_api requests pandas numpy

# Test data service
python nba_data_service.py

# Test specific endpoints
python nba_data_service.py players '{"limit": 10}'
```

### Testing Strategy
- **Unit Tests**: Individual API endpoint functionality
- **Integration Tests**: Full data pipeline from NBA API to frontend
- **E2E Tests**: Complete user workflows including NBA data usage
- **Performance Tests**: Cache hit rates and response times

## Future Enhancements

### Advanced Features
- **Real-time Alerts**: Live game notifications and betting opportunities
- **Advanced Analytics**: Machine learning models for prediction
- **Custom Datasets**: User-defined data collections and filters
- **API Webhooks**: Real-time data updates via webhooks

### Data Expansion
- **College Basketball**: NCAA data integration
- **International Basketball**: FIBA and international leagues
- **Player Injuries**: Injury reports and availability tracking
- **Advanced Metrics**: PER, true shooting, advanced analytics

---

## Quick Start for Users

1. **Access NBA Data**: Navigate to "NBA Data" tab in dashboard
2. **Browse Players**: Search and filter NBA player roster
3. **View Teams**: Explore team information and performance
4. **Check Live Games**: Monitor real-time game scores
5. **Analyze Stats**: Deep dive into player and team statistics
6. **Build Strategies**: Use data to create sports betting strategies

The NBA integration provides comprehensive sports data to power your backtesting strategies! 🏀📊
